package minesy;

import javax.swing.JOptionPane;
import javax.swing.JDialog;
import javax.swing.JTextField;
import java.beans.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.text.*;

/**
 * This class handles all the stuff of the board like showing current status, catching mouse clicks etc.
*/
class Board extends JPanel implements MouseListener
{
	final int CELL_MINE=1;
	final int CELL_FLAG=2;
	final int CELL_UNSURE=4;
	final int CELL_SHOW=8;
	final int CELL_SIZE=15;
	
	int ROWS, COLS, MINES;
	int cells[][];
	Color c_white, c_ltgray, c_flag;
	boolean gameon;

//frame functions

	/**
	 * Main constructor, just a stub for the init function
	 * @param x width of board
	 * @param y height of board
	 * @param m number of mines
	 * @see init
	 * @see newGame
	 */
	Board(int x, int y, int m) {
		init(x, y, m);
	}
	
	/**
	 * Initializes the board object - creates required objects, sets up the JFrame, starts the game by calling newGame
	 * @param x width of board
	 * @param y height of board
	 * @param m number of mines
	 * @see newGame
	 */
	public void init(int x, int y, int m) {
		c_white=new Color((float)1.0, (float)1.0, (float)1.0);
		c_ltgray=new Color((float)0.90, (float)0.90, (float)0.90);
		c_flag=new Color((float)0.99, (float)0.80, (float)0.80);
		
		addMouseListener(this);
		
		newGame(x, y, m);
	}
	
	/**
	 * Returns preferred size of the frame (size that is required to show all fields of the board
	 */
	public Dimension getPreferredSize() {
		//JOptionPane.showMessageDialog(this, "Preferred size: " + COLS*CELL_SIZE+1 + "x" + ROWS*CELL_SIZE+1);

		return new Dimension(COLS*CELL_SIZE+1, ROWS*CELL_SIZE+1);
	}
	
	/**
	 * Draws the plaing board
	 * @param g Graphics object the board will be drawn to
	 */
	public void paint(Graphics g) {
		int x, y, i;

		//draw grid
		for (y=0; y<ROWS+1; y++) {
			g.drawLine(0, y*CELL_SIZE, COLS*CELL_SIZE, y*CELL_SIZE);
		}

		for (x=0; x<COLS+1; x++) {
			g.drawLine(x*CELL_SIZE, 0, x*CELL_SIZE, ROWS*CELL_SIZE);
		}
		
		//draw cells
		for (y=0; y<ROWS; y++) {
			for (x=0; x<COLS; x++) {
				if ((cells[y][x]&CELL_SHOW) == CELL_SHOW)
					g.setColor(c_white);
				else if (((cells[y][x]&CELL_FLAG) == CELL_FLAG) || ((cells[y][x]&CELL_UNSURE) == CELL_UNSURE))
					g.setColor(c_flag);
				else g.setColor(c_ltgray);
				
				g.fillRect(x*CELL_SIZE+1, y*CELL_SIZE+1, CELL_SIZE-1, CELL_SIZE-1);
				
				if (!gameon && (cells[y][x]&CELL_MINE) == CELL_MINE) {
					g.setColor(Color.red);
					g.drawString("@", x*CELL_SIZE+3, y*CELL_SIZE+CELL_SIZE-2);
				}
				if ((cells[y][x]&CELL_FLAG) == CELL_FLAG) {
					g.setColor(Color.blue);
					g.drawString("#", x*CELL_SIZE+3, y*CELL_SIZE+CELL_SIZE-2);
				}
				if ((cells[y][x]&CELL_UNSURE) == CELL_UNSURE) {
					g.setColor(Color.blue);
					g.drawString("?", x*CELL_SIZE+3, y*CELL_SIZE+CELL_SIZE-2);
				}
				
				g.setColor(Color.black);
				if ((cells[y][x]&CELL_SHOW) == CELL_SHOW) {
					i=minesAround(x, y);
					if (i!=0)
						g.drawString(""+i, x*CELL_SIZE+3, y*CELL_SIZE+CELL_SIZE-2);
				}
			}
		}
	}

	/**
	 * Called when user presses a mouse button
	 * @param e event specification
	 */
	public void mousePressed(MouseEvent e) {
	}

	/**
	 * Called when user releases a mouse button
	 * @param e event specification
	 */
	public void mouseReleased(MouseEvent e) {
	}
	
	/**
	 * Called when the game is over to let the user know why it ended
	 * @param how if it's 0, user lost; if it's 1, user won
	 */
	public void gameOver(int how) {
		gameon=false;
		repaint();
		if (how==1)
			JOptionPane.showMessageDialog(this, "Congrats, you won! ;-)");
		else JOptionPane.showMessageDialog(this, "You are a looser, better try it again :-P");
	}
	
	/**
	 * Called when user clicks a mouse button
	 * It tries to explore the board, puts flags, terminates the game when user clicks a mine
	 * @param e event specification
	 */
	public void mouseClicked(MouseEvent e) {
	    if (!gameon)
	        return;
	    
		int x = e.getX();
		int y = e.getY();
		int b= e.getButton();
		
		if (((x<(COLS+1)*CELL_SIZE)) && (y<(ROWS+1)*CELL_SIZE) && (x>=0) && (y>=0)) {
			x=(int)(x/CELL_SIZE);
			y=(int)(y/CELL_SIZE);
			
			switch (b) {
			case MouseEvent.BUTTON1: //left
				switch (explore(x, y)) {
				case 0: //ok
					if (isWon())
						gameOver(1);
					else repaint();
					break;
				case 1: //mine
					gameOver(0);
					break;
				}
				break;
			case MouseEvent.BUTTON3: //right
				if ((cells[y][x]&(CELL_FLAG|CELL_UNSURE|CELL_SHOW)) == 0) {
					cells[y][x]|=CELL_FLAG;
					repaint();
				} else {
					if ((cells[y][x]&CELL_FLAG) == CELL_FLAG) {
						cells[y][x]=(cells[y][x]& ~CELL_FLAG)|CELL_UNSURE;
						repaint();
					} else if ((cells[y][x]&CELL_UNSURE) == CELL_UNSURE) {
						cells[y][x]=(cells[y][x]& ~CELL_UNSURE);
						repaint();
					}
				}
				break;
			}
		}
	}

	/**
	 * Called when user mouse enters board's region
	 * @param e event specification
	 */
	public void mouseEntered(MouseEvent e) {
	}

	/**
	 * Called when user mouse leaves board's region
	 * @param e event specification
	 */
	public void mouseExited(MouseEvent e) {
	}
	
//game functions

	/**
	 * Starts a new game - creates board, and repaints it
	 * @param x width of board
	 * @param y height of board
	 * @param n number of mines
	 * @see newBoard
	 */
	public void newGame(int w, int h, int n) {
	    newBoard(w, h, n);
	    gameon=true;
	    repaint();
	}
	
	/**
	 * Creates board of specified parameters
	 * @param w width of board
	 * @param h height of board
	 * @param n number of mines
	 */
	private void newBoard(int w, int h, int n) {
		int x, y, i;
		
		ROWS=w;
		COLS=h;
		MINES=n;
		
		cells=new int[ROWS][COLS];
		for (y=0; y<ROWS; y++) 
			for (x=0; x<COLS; x++)
				cells[y][x]=0;
		for (i=0; i<MINES;) {
			x=(int)(Math.random()*COLS);
			y=(int)(Math.random()*ROWS);
			if (cells[y][x]==0) {
				cells[y][x]=CELL_MINE;
				i++;
			}
		}
	}
	
	/**
	 * Returns true, if [x,y] is inside the board's dimensions
	 * @param x width of board
	 * @param y height of board
	 */
	private boolean inCells(int x, int y) {
		return (x>=0 && y>=0 && x<COLS && y<ROWS);
	}
	
	/**
	 * Returns true, if a mine is at [x,y]
	 * @param x width of board
	 * @param y height of board
	 */
	private int isMine(int x, int y) {
		if (!inCells(x, y)) return 0;
		else return ((cells[y][x]&CELL_MINE)==CELL_MINE)?1:0;
	}
	
	/**
	 * Returns count of mines arount [x,y]
	 * @param x width of board
	 * @param y height of board
	 */
	private int minesAround(int x, int y) {
		return
			isMine(x-1, y-1)+isMine(x-1, y)+isMine(x-1, y+1)+
			isMine(x, y-1)+isMine(x, y)+isMine(x, y+1)+
			isMine(x+1, y-1)+isMine(x+1, y)+isMine(x+1, y+1);
	}
	
	/**
	 * Explores board from the position [x,y]
	 * Sets [x,y] to CELL_SHOW, and tries to explore the fields nearby
	 * Returns 1 if a place with mine should have been explored, 0 otherwise
	 * @param x width of board
	 * @param y height of board
	 */
	private int explore(int x, int y) {
		if (!inCells(x, y) || (cells[y][x]&CELL_SHOW)==CELL_SHOW) {
			return 0;
		}

		if ((cells[y][x]&CELL_MINE)==CELL_MINE) {
			cells[y][x]=CELL_SHOW|CELL_MINE;
			return 1;
		}
		cells[y][x]=CELL_SHOW;
		if (minesAround(x, y)==0) {
			explore(x-1, y-1);
			explore(x+1, y+1);
			explore(x+1, y-1);
			explore(x-1, y+1);
			
			explore(x-1, y);
			explore(x+1, y);
			explore(x, y-1);
			explore(x, y+1);
		}
		return 0;
	}
	
	/**
	 * Returns true, if the game is won (i.e. number of fields not shown is equal to number of mines); false otherwise
	 */
	private boolean isWon() {
		int x, y;
		
		for (y=0; y<ROWS; y++) {
			for (x=0; x<COLS; x++)
				if ((cells[y][x]&CELL_MINE)!=CELL_MINE && (cells[y][x]&CELL_SHOW)!=CELL_SHOW)
					return false;
		}
		return true;
	}
}

/**
 * Settings dialog for the game
 */
class SettingsDialog extends JDialog implements ActionListener {
	public int x, y, n;
	private DecimalField fx, fy, fn;
	private NumberFormat nf;
	JButton b1, b2, b3, bok, bcan;
	
	/**
	 * Creates the dialog's controls and sets _x, _y, _n as initial values
	 * @param aFrame parent frame
	 * @param _x initial width
	 * @param _y initial height
	 * @param _n initial number of mines
	 */
	SettingsDialog(Frame aFrame, int _x, int _y, int _n) {
		super(aFrame, true);
		
		x=_x; y=_y; n=_n;
		
		nf=NumberFormat.getNumberInstance();
	
		setTitle("Settings");
		setSize(500,400);
		setResizable(false);
		
		setLayout(new GraphPaperLayout(new Dimension(3,7)));

		add(new JLabel("x size:"), new Rectangle(0,0,1,1));
		fx=new DecimalField(x, 3, nf);
		add(fx, new Rectangle(1,0,1,1));
	
		add(new JLabel("y size:"), new Rectangle(0,1,1,1));
		fy=new DecimalField(y, 3, nf);
		add(fy, new Rectangle(1,1,1,1));
	
		add(new JLabel("no mines:"), new Rectangle(0,2,1,1));
		fn=new DecimalField(n, 3, nf);
		add(fn, new Rectangle(1,2,1,1));

		b1=new JButton("Beginner");
		b1.setActionCommand("lbeginner"); 
		b2=new JButton("Normal");
		b2.setActionCommand("lnormal"); 
		b3=new JButton("Masta");
		b3.setActionCommand("lmasta"); 
		
		bok=new JButton("OK");
		bok.setActionCommand("ok"); 
		bcan=new JButton("Cancel");
		bcan.setActionCommand("cancel"); 
		b1.addActionListener(this);
		b2.addActionListener(this);
		b3.addActionListener(this);
		bok.addActionListener(this);
		bcan.addActionListener(this);
		
		add(new JLabel(" "), new Rectangle(0, 3, 1, 1));
		add(b1, new Rectangle(0, 4, 1, 1));
		add(b2, new Rectangle(1, 4, 1, 1));
		add(b3, new Rectangle(2, 4, 1, 1));
		add(new JLabel(" "), new Rectangle(0, 5, 1, 1));
		add(bok, new Rectangle(0, 6, 1, 1));
		add(bcan, new Rectangle(1, 6, 1, 1));
		pack();
		setVisible(true);
	}
	
	/**
	 * Standard function - processes events like WINDOW_CLOSING etc.
	 * @param e event sent to window
	 */
	public void processWindowEvent(WindowEvent e) {
		if(e.getID() == WindowEvent.WINDOW_CLOSING)
			setVisible(false);
	}
	
	/**
	 * Standard function - processes button clicks
	 * @param e action performed
	 */
	public void actionPerformed(ActionEvent e) {
		String c = e.getActionCommand();
		if (c.equals("lbeginner")) {
			fx.setValue(10);
			fy.setValue(10);
			fn.setValue(10);
		} else if (c.equals("lnormal")) {
			fx.setValue(15);
			fy.setValue(15);
			fn.setValue(30);
		} else if (c.equals("lmasta")) {
			fx.setValue(30);
			fy.setValue(30);
			fn.setValue(200);
		} else if (c.equals("ok")) {
			x=(int)fx.getValue();
			y=(int)fy.getValue();
			n=(int)fn.getValue();
			setVisible(false);
		} else if (c.equals("cancel")) {
			setVisible(false);
		}
	}
}